<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\TherapistCategoriesExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\TherapistCategories\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\TherapistCategoriesImport;
use App\Models\TherapistCategory;
use App\Models\TherapistMainCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;

class TherapistCategoriesController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist_category.index');
        $this->middleware('permission:therapist_category.add', ['only' => ['store']]);
        $this->middleware('permission:therapist_category.edit', ['only' => ['update']]);
        $this->middleware('permission:therapist_category.delete', ['only' => ['destroy']]);
        $this->middleware('permission:therapist_category.export', ['only' => ['export']]);
        $this->middleware('permission:therapist_category.import', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null)
    {
        if ($req != null) {
            $therapist_categories =  TherapistCategory::withAll();
            if ($req->trash && $req->trash == 'with') {
                $therapist_categories =  $therapist_categories->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_categories =  $therapist_categories->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_categories = $therapist_categories->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_categories = $therapist_categories->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_categories = $therapist_categories->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_categories = $therapist_categories->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_categories = $therapist_categories->get();
                return $therapist_categories;
            }
            $therapist_categories = $therapist_categories->get();
            return $therapist_categories;
        }
        $therapist_categories = TherapistCategory::withAll()->orderBy('id', 'desc')->get();
        return $therapist_categories;
    }


    /*********View All TherapistCategories  ***********/
    public function index(Request $request)
    {
        $therapist_categories = $this->getter($request);
        return view('super_admins.therapist_categories.index')->with('therapist_categories', $therapist_categories);
    }

    /*********View Create Form of TherapistCategory  ***********/
    public function create()
    {
        $therapist_main_categories = TherapistMainCategory::active()->get();

        return view('super_admins.therapist_categories.create', compact('therapist_main_categories'));
    }

    /*********Store TherapistCategory  ***********/
    public function store(CreateRequest $request)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $data['image'] = uploadCroppedFile($request, 'image', 'therapist_categories');

            $therapist_category = TherapistCategory::create($data);
            $therapist_category->slug = Str::slug($therapist_category->name . ' ' . $therapist_category->id, '-');
            $therapist_category->save();
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_categories.index')->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_categories.index')->with('message', 'TherapistCategory Created Successfully')->with('message_type', 'success');
    }

    /*********View TherapistCategory  ***********/
    public function show(TherapistCategory $therapist_category)
    {
        return view('super_admins.therapist_categories.show', compact('therapist_category'));
    }

    /*********View Edit Form of TherapistCategory  ***********/
    public function edit(TherapistCategory $therapist_category)
    {
        $therapist_main_categories = TherapistMainCategory::active()->get();

        return view('super_admins.therapist_categories.edit', compact('therapist_category', 'therapist_main_categories'));
    }

    /*********Update TherapistCategory  ***********/
    public function update(CreateRequest $request, TherapistCategory $therapist_category)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            if ($request->image) {
                $data['image'] = uploadCroppedFile($request, 'image', 'therapist_categories', $therapist_category->image);
            } else {
                $data['image'] = $therapist_category->image;
            }
            $therapist_category->update($data);
            $therapist_category = TherapistCategory::find($therapist_category->id);
            $slug = Str::slug($therapist_category->name . ' ' . $therapist_category->id, '-');
            $therapist_category->update([
                'slug' => $slug
            ]);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_categories.index')->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_categories.index')->with('message', 'TherapistCategory Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_categories = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_categories." . $extension;
        return Excel::download(new TherapistCategoriesExport($therapist_categories), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new TherapistCategoriesImport, $file);
        return redirect()->back()->with('message', 'Blog Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE TherapistCategory ***********/
    public function destroy(TherapistCategory $therapist_category)
    {
        $therapist_category->delete();
        return redirect()->back()->with('message', 'TherapistCategory Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE TherapistCategory ***********/
    public function destroyPermanently(Request $request, $therapist_category)
    {
        $therapist_category = TherapistCategory::withTrashed()->find($therapist_category);
        if ($therapist_category) {
            if ($therapist_category->trashed()) {
                if ($therapist_category->image && file_exists(public_path($therapist_category->image))) {
                    unlink(public_path($therapist_category->image));
                }
                $therapist_category->forceDelete();
                return redirect()->back()->with('message', 'Blog Category Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'Blog Category is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'Blog Category Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore TherapistCategory***********/
    public function restore(Request $request, $therapist_category)
    {
        $therapist_category = TherapistCategory::withTrashed()->find($therapist_category);
        if ($therapist_category->trashed()) {
            $therapist_category->restore();
            return redirect()->back()->with('message', 'Blog Category Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'Blog Category Not Found')->with('message_type', 'error');
        }
    }
}
